<?php

namespace Drupal\Tests\layout_builder_lock\Functional;

use Behat\Mink\Element\NodeElement;
use Drupal\block_content\Entity\BlockContentType;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay;
use Drupal\layout_builder_lock\LayoutBuilderLock;
use Drupal\node\NodeInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\WebAssert;

/**
 * Tests Layout Builder Lock.
 *
 * @group layout_builder_lock
 */
class LayoutBuilderLockTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'layout_builder',
    'layout_builder_lock',
    'block_content',
    'block',
    'node',
    'field_ui',
    'user',
  ];

  /**
   * The body field uuid.
   *
   * @var string
   */
  protected $bodyFieldBlockUuid;

  /**
   * The custom default block uuid.
   *
   * @var string
   */
  protected $customDefaultBlockUuid;

  /**
   * The editor block UUID.
   *
   * @var string
   */
  protected $customEditorBlockUuid;


  /**
   * The default theme to use.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with all permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A user with all permissions except bypass.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUserNoBypass;

  /**
   * A user with default permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $editor;

  /**
   * A user which can override lock settings overrides.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $editorOverride;

  /**
   * A user which can remove sections with lock settings.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $editorRemoveSection;

  /**
   * The default editor permissions.
   *
   * @var array
   */
  protected $editorPermissions = [
    'bypass node access',
    'configure any layout',
    'create and edit custom blocks',
    'access contextual links',
  ];

  /**
   * The editor permissions.
   *
   * @var array
   */
  protected $editorOverridePermissions = [
    'bypass node access',
    'configure any layout',
    'create and edit custom blocks',
    'access contextual links',
    'manage lock settings on overrides',
  ];

  /**
   * The editor permissions to remove sections with lock settings.
   *
   * @var array
   */
  protected $editorRemoveSectionPermissions = [
    'bypass node access',
    'configure any layout',
    'create and edit custom blocks',
    'access contextual links',
    'remove sections with lock settings',
  ];

  /**
   * The editor permissions.
   *
   * @var array
   */
  protected $adminUserNoBypassPermissions = [
    'bypass node access',
    'configure any layout',
    'create and edit custom blocks',
    'access contextual links',
    'administer node display',
    'manage lock settings on sections',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Enable Layout Builder for landing page.
    $this->createContentType(['type' => 'landing_page']);

    $bundle = BlockContentType::create([
      'id' => 'basic',
      'label' => 'Basic',
      'revision' => FALSE,
    ]);
    $bundle->save();
    block_content_add_body_field($bundle->id());

    LayoutBuilderEntityViewDisplay::load('node.landing_page.default')
      ->enableLayoutBuilder()
      ->setOverridable()
      ->save();

    try {
      $this->adminUser = $this->createUser([], 'administrator', TRUE);
      $this->adminUserNoBypass = $this->createUser($this->adminUserNoBypassPermissions, 'administratorNoByPass');
      $this->editor = $this->createUser($this->editorPermissions, 'editor');
      $this->editorOverride = $this->createUser($this->editorOverridePermissions, 'editorOverride');
      $this->editorRemoveSection = $this->createUser($this->editorRemoveSectionPermissions, 'editorRemoveSection');
    }
    catch (EntityStorageException $exception) {
      return;
    }
  }

  /**
   * Tests locking features on sections.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testLock(): void {
    $assert_session = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Create first node.
    $node_1 = $this->drupalCreateNode([
      'type' => 'landing_page',
      'title' => 'Homepage 1',
    ]);

    // Check as editor.
    $this->drupalLogin($this->editor);
    $this->drupalGet('node/' . $node_1->id() . '/layout');

    // Get the block uuid from the body field.
    $id = $assert_session->elementExists('css', '.layout-builder__region > div:nth-child(3)');
    $this->bodyFieldBlockUuid = $id->getAttribute('data-layout-block-uuid');

    // Check links and access.
    $this->checkLinksAndAccess($assert_session, $node_1);

    // Configure the section locks.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/layout_builder/configure/section/defaults/node.landing_page.default/0');

    $edit = [];
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_BLOCK_ADD . ']'] = TRUE;
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_BLOCK_MOVE . ']'] = TRUE;
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_BLOCK_UPDATE . ']'] = TRUE;
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_BLOCK_DELETE . ']'] = TRUE;
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_BEFORE . ']'] = TRUE;
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_BLOCK_MOVE . ']'] = TRUE;
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_AFTER . ']'] = TRUE;
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_CONFIGURE . ']'] = TRUE;
    $this->submitForm($edit, 'Update');
    $page->pressButton('Save layout');

    // Create second node.
    $node_2 = $this->drupalCreateNode([
      'type' => 'landing_page',
      'title' => 'Homepage 2',
    ]);

    // Check as editor.
    $this->drupalLogin($this->editor);
    $this->drupalGet('node/' . $node_2->id() . '/layout');
    $this->checkLinksAndAccess($assert_session, $node_2, TRUE, 403);

    // Links will still exist on node 1 as the overridden settings are used.
    $this->drupalGet('node/' . $node_1->id() . '/layout');
    $this->checkLinksAndAccess($assert_session, $node_1);

    // Override per entity is allowed for administrators.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('layout_builder/configure/section/overrides/node.' . $node_2->id() . '/0');
    $assert_session->statusCodeEquals(200);
    $assert_session->responseContains('Lock settings');

    // Check if a user can manage override.
    $this->drupalLogin($this->editor);
    $this->drupalGet('node/' . $node_2->id() . '/layout');
    $assert_session->responseNotContains('Configure section 1');
    $this->drupalGet('layout_builder/configure/section/overrides/node.' . $node_2->id() . '/0');
    $assert_session->statusCodeEquals(403);

    $this->drupalLogin($this->editorOverride);
    $this->drupalGet('node/' . $node_2->id() . '/layout');
    $assert_session->responseContains('Configure section 1');
    $this->drupalGet('layout_builder/configure/section/overrides/node.' . $node_2->id() . '/0');
    $assert_session->statusCodeEquals(200);

    // Override settings on override.
    $node_3 = $this->drupalCreateNode([
      'type' => 'landing_page',
      'title' => 'Homepage 3',
    ]);
    $this->drupalGet('node/' . $node_3->id() . '/layout');
    $assert_session->statusCodeEquals(200);
    $this->drupalGet('layout_builder/configure/section/overrides/node.' . $node_3->id() . '/0');
    $assert_session->statusCodeEquals(200);
    $assert_session->responseContains('Lock settings');
    $edit = [];
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_AFTER . ']'] = FALSE;
    $this->submitForm($edit, 'Update');
    $page->pressButton('Save layout');

    $this->drupalLogin($this->editor);
    $this->drupalGet('node/' . $node_3->id() . '/layout');
    $this->checkLinksAndAccess($assert_session, $node_3, TRUE, 403, TRUE);

    // Try to add a new section.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/1/layout_onecol');
    $assert_session->statusCodeEquals(200);
    $assert_session->responseContains('Locks can be configured when the section has been added.');
    $this->submitForm([], 'Add section');
    $assert_session->statusCodeEquals(200);

    // Test the 'bypass lock settings on layout overrides', in combination
    // with 'manage lock settings on sections'. In this case, the user does not
    // have the permission to do anything on the override.
    $this->drupalLogin($this->adminUserNoBypass);
    $this->drupalGet('/layout_builder/configure/section/defaults/node.landing_page.default/0');
    $assert_session->statusCodeEquals(200);
    $assert_session->responseContains('Lock settings');
    $this->drupalGet('node/' . $node_2->id() . '/layout');
    $this->checkLinksAndAccess($assert_session, $node_2, TRUE, 403);

    // Check custom inline block can be updated in a section that is configured
    // to allow adding new blocks and not allowing updating default blocks.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/layout_builder/configure/section/defaults/node.landing_page.default/0');
    $edit = [];
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_BLOCK_ADD . ']'] = FALSE;
    $this->submitForm($edit, 'Update');
    $this->drupalGet('/layout_builder/add/block/defaults/node.landing_page.default/0/content/inline_block:basic');
    $edit = [];
    $edit['settings[label]'] = 'Default custom block title';
    $edit['settings[block_form][body][0][value]'] = 'Default custom block content';
    $this->submitForm($edit, 'Add block');

    // Get the block uuid from the custom block.
    $id = $assert_session->elementExists('css', '.layout-builder__region > div:nth-child(4)');
    $this->customDefaultBlockUuid = $id->getAttribute('data-layout-block-uuid');
    $page->pressButton('Save layout');
    $this->drupalGet('/layout_builder/configure/section/defaults/node.landing_page.default/0');
    $assert_session->checkboxNotChecked('layout_builder_lock[4]');
    $this->drupalGet('/layout_builder/update/block/defaults/node.landing_page.default/0/content/' . $this->customDefaultBlockUuid);
    $assert_session->statusCodeEquals(200);
    $assert_session->responseContains('Default custom block content');

    // Check as editor.
    $node_4 = $this->drupalCreateNode([
      'type' => 'landing_page',
      'title' => 'Landing page 2',
    ]);
    $this->drupalLogin($this->editor);
    $this->drupalGet('node/' . $node_4->id() . '/layout');
    $assert_session->responseContains('Default custom block content');
    $assert_session->linkExists('Add block');
    $this->drupalGet('/layout_builder/update/block/overrides/node.' . $node_4->id() . '/0/content/' . $this->customDefaultBlockUuid);
    $assert_session->statusCodeEquals(403);

    // Add custom block as editor.
    $this->drupalGet('/layout_builder/add/block/overrides/node.' . $node_4->id() . '/0/content/inline_block:basic');
    $edit = [];
    $edit['settings[label]'] = 'Editor block title';
    $edit['settings[block_form][body][0][value]'] = 'Editor block content';
    $this->submitForm($edit, 'Add block');
    $id = $assert_session->elementExists('css', '.layout-builder__region > div:nth-child(5)');
    $this->customEditorBlockUuid = $id->getAttribute('data-layout-block-uuid');
    $page->pressButton('Save layout');
    $assert_session->responseContains('Editor block content');
    $this->drupalGet('/layout_builder/update/block/overrides/node.' . $node_4->id() . '/0/content/' . $this->customEditorBlockUuid);
    $assert_session->statusCodeEquals(200);
  }

  /**
   * Tests removing a section with the permission to do so.
   */
  public function testRemovingSectionWithAppropriatePermission(): void {
    $assert_session = $this->assertSession();
    $page = $this->getSession()->getPage();
    $node = $this->drupalCreateNode([
      'type' => 'landing_page',
      'title' => 'Homepage 1',
    ]);

    // Login As a user with the permission "Bypass lock settings on layout
    // overrides".
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/layout_builder/configure/section/defaults/node.landing_page.default/0');

    // Set any layout_builder_lock settings and save the section.
    $edit['layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_CONFIGURE . ']'] = TRUE;
    $this->submitForm($edit, 'Update');
    $page->pressButton('Save layout');

    // Login as a user without the permission "Bypass lock settings on layout
    // overrides".
    $this->drupalLogin($this->editorRemoveSection);
    $this->drupalGet('node/' . $node->id() . '/layout');
    // The link to remove the section should be present.
    $assert_session->linkExists('Remove Section 1');
    // It's possible to remove the section via the link as well.
    $this->drupalGet('/layout_builder/remove/section/overrides/node.' . $node->id() . '/0');
    $assert_session->statusCodeEquals(200);
  }

  /**
   * Tests with at least 3 sections.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testMultipleSections(): void {
    $assert_session = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Ensure the lock settings are added to the correct section when adding a
    // new section before an existing section. We expect that:
    // - we can't add lock settings for new sections.
    // @see https://www.drupal.org/project/layout_builder_lock/issues/3129009;
    // - the settings are stored on the correct section;
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/0/layout_onecol');
    $assert_session->responseContains('Locks can be configured when the section has been added.');

    // Add a new section above default.
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/0/layout_onecol');
    $this->submitForm(['layout_settings[label]' => 'section above default'], 'Add section');
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/0');
    $this->submitForm(['layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_CONFIGURE . ']' => TRUE], 'Update');

    // Add a new section between previous created and default.
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/1/layout_onecol');
    $this->submitForm(['layout_settings[label]' => 'new section in between'], 'Add section');
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/1');
    $this->submitForm(['layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_BEFORE . ']' => TRUE], 'Update');

    $page->pressButton('Save layout');

    // We expect that the first section (first added) has the
    // `locked section configure` checkbox checked.
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/0');
    $assert_session->checkboxChecked('layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_CONFIGURE . ']');
    $assert_session->checkboxNotChecked('layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_BEFORE . ']');

    // We expect that the second section (last added) has the
    // `locked section before` checkbox checked.
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/1');
    $assert_session->checkboxChecked('layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_BEFORE . ']');
    $assert_session->checkboxNotChecked('layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_CONFIGURE . ']');

    // We expect that the third section (default) has no checkboxes checked.
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/2');
    $assert_session->checkboxNotChecked('layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_CONFIGURE . ']');
    $assert_session->checkboxNotChecked('layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_BEFORE . ']');

    // Create a node.
    $node = $this->drupalCreateNode([
      'type' => 'landing_page',
      'title' => 'Homepage',
    ]);

    // Simply login as an editor. Should not throw any PHP error.
    // @see https://www.drupal.org/project/layout_builder_lock/issues/3121250
    $this->drupalLogin($this->editor);
    $this->drupalGet('node/' . $node->id() . '/layout');

    $assert_session->linkByHrefExists('/layout_builder/choose/section/overrides/node.' . $node->id() . '/0');
    $assert_session->linkByHrefNotExists('/layout_builder/choose/section/overrides/node.' . $node->id() . '/1');
    $assert_session->linkByHrefExists('/layout_builder/choose/section/overrides/node.' . $node->id() . '/2');

    $assert_session->linkExists('Configure new section in between');
    $assert_session->linkExists('Configure Section 3');

    // Ensure sections with empty lock config don't mess up the subsequent
    // 'configure section' links.
    $this->drupalLogin($this->adminUser);

    // Add a section without config so it has a section delta > 1.
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/2/layout_onecol');
    $this->submitForm(['layout_settings[label]' => 'section without any lock config'], 'Add section');

    // Add extra sections that have section configuration locked.
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/3/layout_onecol');
    $this->submitForm(['layout_settings[label]' => 'section with locked section configuration  1'], 'Add section');
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/3');
    $this->submitForm([
      'layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_CONFIGURE . ']' => TRUE,
      'layout_settings[label]' => 'section with locked section configuration  1',
    ], 'Update');

    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/5/layout_onecol');
    $this->submitForm(['layout_settings[label]' => 'section with locked section configuration  2'], 'Add section');
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/5');
    $this->submitForm([
      'layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_CONFIGURE . ']' => TRUE,
      'layout_settings[label]' => 'section with locked section configuration  2',
    ], 'Update');

    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/6/layout_onecol');
    $this->submitForm(['layout_settings[label]' => 'section with locked section configuration  3'], 'Add section');
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/6');
    $this->submitForm([
      'layout_builder_lock[' . LayoutBuilderLock::LOCKED_SECTION_CONFIGURE . ']' => TRUE,
      'layout_settings[label]' => 'section with locked section configuration 3',
    ], 'Update');

    $page->pressButton('Save layout');

    // Create a node.
    $node = $this->drupalCreateNode([
      'type' => 'landing_page',
      'title' => 'Homepage',
    ]);

    $this->drupalLogin($this->editor);
    $this->drupalGet('node/' . $node->id() . '/layout');

    $assert_session->linkNotExists('Configure section above default');
    $assert_session->linkExists('Configure new section in between');
    $assert_session->linkExists('Configure section without any lock config');
    $assert_session->linkNotExists('Configure section with locked configure 1');
    $assert_session->linkExists('Configure Section 5');
    $assert_session->linkNotExists('Configure section with locked configure 2');
    $assert_session->linkNotExists('Configure section with locked configure 3');
  }

  /**
   * Tests region locking.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testRegionLock(): void {
    $assertSession = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Log in as admin and add a section with a two-column layout.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/1/layout_twocol');
    $this->submitForm([], 'Add section');

    // Add a block to each column.
    $this->drupalGet('layout_builder/add/block/defaults/node.landing_page.default/1/first/inline_block:basic');
    $this->submitForm([
      'settings[label]' => 'First block title',
      'settings[block_form][body][0][value]' => 'First block content',
    ], 'Add block');
    $this->drupalGet('layout_builder/add/block/defaults/node.landing_page.default/1/second/inline_block:basic');
    $this->submitForm([
      'settings[label]' => 'Second block title',
      'settings[block_form][body][0][value]' => 'Second block content',
    ], 'Add block');

    // Lock the second region and save the layout.
    $this->drupalGet('layout_builder/configure/section/defaults/node.landing_page.default/1');
    $this->submitForm([
      'layout_builder_lock_regions[second]' => TRUE,
    ], 'Update');
    $page->pressButton('Save layout');

    // Create a node and get the UUIDs for both blocks.
    $node = $this->drupalCreateNode([
      'type' => 'landing_page',
      'title' => 'Region lock test page',
    ]);
    $this->drupalGet('node/' . $node->id() . '/layout');
    $blockIds = [];
    foreach (['first', 'second'] as $region) {
      $block = $assertSession->elementExists('css', '.layout-builder__region.layout__region--' . $region . ' > .layout-builder-block');
      $blockIds[$region] = $block->getAttribute('data-layout-block-uuid');
    }

    // Confirm that the admin can still manage blocks in both regions.
    foreach ($blockIds as $region => $blockId) {
      $this->checkBlockRouteAccess($assertSession, $node, 200, 1, $region, $blockId);

      // Check contextual links.
      $this->drupalGet('node/' . $node->id() . '/layout');
      $block = $assertSession->elementExists('css', '.layout-builder__region.layout__region--' . $region . ' > .layout-builder-block');
      $links = $block->find('xpath', '//div[@data-contextual-id]');
      $this->checkContextualLinks($links);
    }

    // Check that the admin can still move blocks to both regions.
    $this->drupalGet('layout_builder/move/block/overrides/node.' . $node->id() . '/1/first/' . $blockIds['first']);
    $options = $assertSession->selectExists('region')->findAll('css', 'option');
    $options = array_map(function ($item) {
      return $item->getValue();
    }, $options);
    $this->assertContains('1:first', $options);
    $this->assertContains('1:second', $options);

    // Log in as editor.
    $this->drupalLogin($this->editor);

    // Confirm that the editor can still manage blocks in the first region.
    $this->checkBlockRouteAccess($assertSession, $node, 200, 1, 'first', $blockIds['first']);
    // Check contextual links.
    $this->drupalGet('node/' . $node->id() . '/layout');
    $block = $assertSession->elementExists('css', '.layout-builder__region.layout__region--first > .layout-builder-block');
    $links = $block->find('xpath', '//div[@data-contextual-id]');
    $this->checkContextualLinks($links);

    // Confirm that the editor cannot manage blocks in the locked second region.
    $this->checkBlockRouteAccess($assertSession, $node, 403, 1, 'second', $blockIds['second']);
    // Check contextual links.
    $this->drupalGet('node/' . $node->id() . '/layout');
    $block = $assertSession->elementExists('css', '.layout-builder__region.layout__region--second > .layout-builder-block-locked');
    $links = $block->find('xpath', '//div[@data-contextual-id]');
    $this->checkContextualLinks($links, TRUE);

    // Confirm that the editor can move blocks to the first region, but not the
    // second region.
    $this->drupalGet('layout_builder/move/block/overrides/node.' . $node->id() . '/1/first/' . $blockIds['first']);
    $options = $assertSession->selectExists('region')->findAll('css', 'option');
    $options = array_map(function ($item) {
      return $item->getValue();
    }, $options);
    $this->assertContains('1:first', $options);
    $this->assertNotContains('1:second', $options);
  }

  /**
   * Checks links and access.
   *
   * @param \Drupal\Tests\WebAssert $assert_session
   *   The WebAssert object used for making assertions in the test.
   * @param \Drupal\node\NodeInterface $node
   *   The Node entity for which links and access are being checked.
   * @param bool $locked
   *   A boolean indicating whether the links are locked or not.
   * @param int $code
   *   Response status code.
   * @param bool $allow_section_after
   *   Flag to allow section after. TRUE to allow, FALSE otherwise.
   *
   * @throws \Behat\Mink\Exception\ElementNotFoundException
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  protected function checkLinksAndAccess(WebAssert $assert_session, NodeInterface $node, bool $locked = FALSE, int $code = 200, bool $allow_section_after = FALSE): void {
    if ($code == 200) {
      $assert_session->linkExists('Add block');
      $assert_session->linkExists('Add section');
      $assert_session->linkExists('Remove Section 1');
      $assert_session->linkExists('Configure Section 1');
      $assert_session->responseContains('js-layout-builder-block');
      $assert_session->responseContains('js-layout-builder-region');
    }
    else {
      if ($allow_section_after) {
        $assert_session->linkExists('Add section');
      }
      else {
        $assert_session->linkNotExists('Add section');
      }
      $assert_session->linkNotExists('Add block');
      $assert_session->linkNotExists('Remove Section 1');
      $assert_session->linkNotExists('Configure Section 1');
      $assert_session->responseNotContains('js-layout-builder-block');
      $assert_session->responseNotContains('js-layout-builder-region');
    }
    $element = $assert_session->elementExists('css', '.layout-builder__region > div:nth-child(3) > div');
    $this->checkContextualLinks($element, $locked);
    $this->checkRouteAccess($assert_session, $node, $code, $allow_section_after);
  }

  /**
   * Checks access to routes related to layout builder.
   *
   * @param \Drupal\Tests\WebAssert $assertSession
   *   The session.
   * @param \Drupal\node\NodeInterface $node
   *   The node.
   * @param int $code
   *   The expected HTTP code.
   * @param bool $sectionAfter
   *   Whether to also check that the next section is accessible.
   * @param int $sectionDelta
   *   The delta of section to check. Default to 0.
   * @param string $region
   *   The region for block access checks. Default to 'content'.
   * @param string|null $blockId
   *   The block UUID to use for block access checks.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  protected function checkRouteAccess(WebAssert $assertSession, NodeInterface $node, int $code = 200, bool $sectionAfter = FALSE, int $sectionDelta = 0, string $region = 'content', ?string $blockId = NULL) {
    $this->checkSectionRouteAccess($assertSession, $node, $code, $sectionAfter, $sectionDelta);
    $this->checkBlockRouteAccess($assertSession, $node, $code, $sectionDelta, $region, $blockId);
  }

  /**
   * Checks access to layout builder section configuration routes.
   *
   * @param \Drupal\Tests\WebAssert $assertSession
   *   The session.
   * @param \Drupal\node\NodeInterface $node
   *   The node.
   * @param int $code
   *   The expected HTTP code.
   * @param bool $sectionAfter
   *   Whether to also check that the next section is accessible.
   * @param int $sectionDelta
   *   The delta of section to check. Default to 0.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  protected function checkSectionRouteAccess(WebAssert $assertSession, NodeInterface $node, int $code = 200, bool $sectionAfter = FALSE, int $sectionDelta = 0) {
    $paths = [
      'layout_builder/configure/section/overrides/node.' . $node->id() . '/' . $sectionDelta,
      'layout_builder/remove/section/overrides/node.' . $node->id() . '/' . $sectionDelta,
      'layout_builder/choose/section/overrides/node.' . $node->id() . '/' . $sectionDelta,
      'layout_builder/choose/section/overrides/node.' . $node->id() . '/' . ($sectionDelta + 1),
    ];
    foreach ($paths as $path) {
      $this->drupalGet($path);

      if ($sectionAfter && $path == 'layout_builder/choose/section/overrides/node.' . $node->id() . '/' . ($sectionDelta + 1)) {
        $assertSession->statusCodeEquals(200);
      }
      else {
        $assertSession->statusCodeEquals($code);
      }
    }
  }

  /**
   * Checks access to layout builder block configuration routes.
   *
   * @param \Drupal\Tests\WebAssert $assertSession
   *   The session.
   * @param \Drupal\node\NodeInterface $node
   *   The node.
   * @param int $code
   *   The expected HTTP code.
   * @param int $sectionDelta
   *   The delta of section to check. Default to 0.
   * @param string $region
   *   The region for block access checks. Default to 'content'.
   * @param string|null $blockId
   *   The block UUID to use for block access checks.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  protected function checkBlockRouteAccess(WebAssert $assertSession, NodeInterface $node, int $code = 200, int $sectionDelta = 0, string $region = 'content', ?string $blockId = NULL) {
    $blockId = $blockId ?? $this->bodyFieldBlockUuid;
    $paths = [
      'layout_builder/choose/block/overrides/node.' . $node->id() . '/' . $sectionDelta . '/' . $region,
      'layout_builder/update/block/overrides/node.' . $node->id() . '/' . $sectionDelta . '/' . $region . '/' . $blockId,
      'layout_builder/move/block/overrides/node.' . $node->id() . '/' . $sectionDelta . '/' . $region . '/' . $blockId,
      'layout_builder/remove/block/overrides/node.' . $node->id() . '/' . $sectionDelta . '/' . $region . '/' . $blockId,
    ];
    foreach ($paths as $path) {
      $this->drupalGet($path);
      $assertSession->statusCodeEquals($code);
    }
  }

  /**
   * Check contextual links.
   *
   * @param \Behat\Mink\Element\NodeElement $links
   *   An element representing the contextual links for a layout builder block.
   * @param bool $locked
   *   Whether the contextual links should be locked.
   */
  protected function checkContextualLinks(NodeElement $links, bool $locked = FALSE) {
    // Parse contextual links.
    $value = $links->getAttribute('data-contextual-id');
    $has_lock_element = FALSE;
    $lockElements = $blockElements = [];

    $elements = explode('&', $value);
    foreach ($elements as $element) {

      // Layout Builder Lock element.
      if (strpos($element, 'layout_builder_lock') !== FALSE) {
        $has_lock_element = TRUE;
        $lockElements = explode(':', str_replace([
          '%3A',
          'layout_builder_lock=',
        ], [':', ''], $element));
      }

      // Layout Builder Block elements.
      if (strpos($element, 'operations') !== FALSE) {
        $exploded = explode(':', $element, 2);
        $block_elements = explode(':', str_replace([
          '%3A',
          'operations=',
        ], [':', ''], $exploded[1]));
      }
    }

    if ($locked) {
      self::assertTrue($has_lock_element);
      self::assertTrue(in_array('layout_builder_block_move', $lockElements));
      self::assertTrue(!in_array('move', $block_elements));
      self::assertTrue(in_array('layout_builder_block_update', $lockElements));
      self::assertTrue(!in_array('update', $block_elements));
      self::assertTrue(in_array('layout_builder_block_remove', $lockElements));
      self::assertTrue(!in_array('remove', $block_elements));
    }
    else {
      self::assertEmpty($has_lock_element);
      self::assertTrue(in_array('move', $block_elements));
      self::assertTrue(in_array('update', $block_elements));
      self::assertTrue(in_array('remove', $block_elements));
    }
  }

}
